import java.util.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * MultipleChoiceQuestions.java 1.0 22/12/12
 * <br><br>
 * Author       : James McMahon
 * <br><br>
 * Provides the Answer Selection Panel that allows the setting of the
 * various elements required.
 * <br><br>
 * These include the setting of the 4 checkboxes and optionally the setting 
 * of the images to go beside them. This class also provides the abilty to determine 
 * which radiobox is actually selected A,B,C or D 
 * <br><br>
 * ////////////////// <br>
 * //Important Info// <br>
 * ////////////////// 
 * <br><br>
 * Radioboxes are used instead of checkboxes because only one answer is ever correct.<br>
 * A change in this structure would need a change in the structure of the input files 
 * and a corresponding change in the Question data structures
 * <br><br>
 * Generally the reason for this implementation over the allowance of multiple answers needed
 * for a correct solution is that the each checkbox has an answer page associated with it.<br>
 * Multiple solutions would break this structure and require two answer pages (minimal) 
 * for each question in a set.<br><br>
 * 
 * ////////////////// <br>
 * //Important Info// <br>
 * //////////////////
 * <br><br>
 *
 * This class does not track which answer is the correct one. <br>
 * It only displays them and returns the one that has been selected when asked.
 * <br><br>
 * Created      : 22/12/2002
 * <br>
 * Last Updated : 22/12/2002
 */

public class AnswerSelectionPanel extends JPanel {
	
	/**
	 * The buttongroup that stops more than one radiobutton being selected at once
	 */
	private ButtonGroup buttonGroup = new ButtonGroup();
	
	/**
	 * The radiobuttons available for display
	 */
	private JRadioButton buttonOne;
	private JRadioButton buttonTwo;
	private JRadioButton buttonThree;
	private JRadioButton buttonFour;

	/** 
	 * The labels used in displaying icons if they are available
	 */
	private JLabel buttonIconLabelOne;
	private JLabel buttonIconLabelTwo;
	private JLabel buttonIconLabelThree;
	private JLabel buttonIconLabelFour;
	
	private Vector defaultCheckBoxText = new Vector(4);
	
	/** 
	 * Constructs the minimal needed for the answer selection panel including default text
	 * for image only checkbox options.
	 * <br>
	 * OTHERS may be added if it is easier to recreate fresh for each question 
	 * as opposed to reseting the available buttons.
	 */
	public AnswerSelectionPanel(){
		
		this.setLayout(new GridLayout(2,4));
		this.setPreferredSize(new Dimension(180,180));
		this.setBorder(BorderFactory.createTitledBorder("Please select an answer"));
		
		//set default text for non text available checkboxes
		defaultCheckBoxText.addElement("A");
		defaultCheckBoxText.addElement("B");
		defaultCheckBoxText.addElement("C");
		defaultCheckBoxText.addElement("D");
	}
	
	/** 
	 * Sets the supplied items in the vector into the checkboxes.
	 * <br>
	 * The vector should contain only the text and should have only four elements
	 * any others are ignored.
	 * 
	 * @param checkBoxText The textual descriptions that are to be set for the checkboxes
	 */
	public void setTextOnlySelectors(Vector checkBoxText) {
		
		removeButtons();//Remove the previous buttons
		
		//Set the buttons
		buttonOne = new JRadioButton((String)checkBoxText.elementAt(0), false);
		buttonTwo = new JRadioButton((String)checkBoxText.elementAt(1), false);						        	  
		buttonThree = new JRadioButton((String)checkBoxText.elementAt(2), false);
		buttonFour = new JRadioButton((String)checkBoxText.elementAt(3), false);
		
		//add the buttons to their screen locations
		this.add(buttonOne); 
		this.add(buttonTwo); 
		this.add(buttonThree);
		this.add(buttonFour);
		
		//add all the buttons to the buttongroup
		buttonGroup.add(buttonOne);
		buttonGroup.add(buttonTwo);
		buttonGroup.add(buttonThree);
		buttonGroup.add(buttonFour);
		
		//Need to remove the icons if they exist because we might have leftovers 
		//from the previous questions
		removeLabels();
		
		this.validate();
		this.repaint();
	}
	
	/** 
	 * Sets the supplied items in the vector into the checkboxes and the images on screen.
	 * <br>
	 * Each vector should contain only the text and images respectively should 
	 * have only four elements, any others are ignored.
	 * 
	 * @param checkBoxText The textual descriptions that are to be set for the checkboxes
	 * @param checkBoxImages The image locations that are to be set for the checkboxes
	 */
	public void setBothSelectors(Vector checkBoxText, Vector checkBoxImages){		
		//Simply calls the other setter methods with the relevant vectors
		setTextOnlySelectors(checkBoxText);//This needs to be called first because it removes all labels
		setImagePlusSelectors(checkBoxImages);	
		
		this.validate();
		this.repaint();		
	}
	
	/** 
	 * Sets the supplied items in the vector beside checkboxes labled A->D.
	 * <br>
	 * The vector should contain only the image urls and should have only four elements
	 * any others are ignored.
	 *
	 * @param checkBoxImages The image locations that are to be set for the checkboxes 
	 */
	public void setImageOnlySelectors(Vector checkBoxImages){
		
		//Remove the previous questions materials
		removeButtons();
		removeLabels();
		
		setTextOnlySelectors(defaultCheckBoxText);
		
		buttonIconLabelOne = new JLabel((ImageIcon)checkBoxImages.elementAt(0), JLabel.LEFT);
		buttonIconLabelTwo = new JLabel((ImageIcon)checkBoxImages.elementAt(1), JLabel.LEFT);
		buttonIconLabelThree = new JLabel((ImageIcon)checkBoxImages.elementAt(2), JLabel.LEFT);
		buttonIconLabelFour = new JLabel((ImageIcon)checkBoxImages.elementAt(3), JLabel.LEFT);
		
		buttonIconLabelOne.setOpaque(true);
		buttonIconLabelTwo.setOpaque(true);
		buttonIconLabelThree.setOpaque(true);
		buttonIconLabelFour.setOpaque(true);
		
		this.add(buttonIconLabelOne);
		this.add(buttonIconLabelTwo);
		this.add(buttonIconLabelThree);
		this.add(buttonIconLabelFour);
		
		this.validate();
		this.repaint();
	}
	
	/** 
	 * Sets the supplied items in the vector beside the checkboxes labled with the proper text
	 * the vector should contain only the image urls and should have only four elements
	 * any others are ignored
	 * 
	 * @param checkBoxImages The image locations that are to be set for the checkboxes 
	 */
	public void setImagePlusSelectors(Vector checkBoxImages){
		
		//just check and ensure no labels are left behind from last question
		removeLabels();
		
		buttonIconLabelOne = new JLabel((ImageIcon)checkBoxImages.elementAt(0), JLabel.LEFT);
		buttonIconLabelTwo = new JLabel((ImageIcon)checkBoxImages.elementAt(1), JLabel.LEFT);
		buttonIconLabelThree = new JLabel((ImageIcon)checkBoxImages.elementAt(2), JLabel.LEFT);
		buttonIconLabelFour = new JLabel((ImageIcon)checkBoxImages.elementAt(3), JLabel.LEFT);
		
		buttonIconLabelOne.setOpaque(true);
		buttonIconLabelTwo.setOpaque(true);
		buttonIconLabelThree.setOpaque(true);
		buttonIconLabelFour.setOpaque(true);
		
		this.add(buttonIconLabelOne);
		this.add(buttonIconLabelTwo);
		this.add(buttonIconLabelThree);
		this.add(buttonIconLabelFour);
	
		this.validate();
		this.repaint();
	}
	
	/** 
	 * Tries to remove all of the labels that are currently on the screen
	 */
	private void removeLabels(){
		try {
			this.remove(buttonIconLabelOne);
			this.remove(buttonIconLabelTwo);
			this.remove(buttonIconLabelThree);
			this.remove(buttonIconLabelFour);
		}catch(NullPointerException npe) {/*The labels didn't need to be removed*/}
		
		this.validate();
		this.repaint();
	}
	
	/** 
	 * Tries to remove all of the buttons that are currently on screen
	 */
	private void removeButtons(){
		try {
			this.remove(buttonOne);
			this.remove(buttonTwo);
			this.remove(buttonThree);
			this.remove(buttonFour);
		}catch(NullPointerException npe) {/*The buttons didn't need to be removed*/}
		
		this.validate();
		this.repaint();
	}
	
	/**
	 * Reset all of the checkboxes to unselected
	 */
	public void resetSelectors() {
			buttonGroup.setSelected(null, true);
	}
	
	/**
	 * Returns the box that is currently selected. <br>
	 * possible values are 0 to 4.<br>
	 * 
	 * @return 0 indicates no checkbox is selected
	 * @return >0 indicates the checkbox that is selected
	 */
	public int getSelectedAnswer(){
		if (buttonOne.isSelected()) {
			return 1;
		}
		else if (buttonTwo.isSelected()) { 
			return 2;
		}
		else if (buttonThree.isSelected()) {
			return 3;
		}
		else if (buttonFour.isSelected()) {
			return 4;
		}
		else {
			return 0;
		}	
	}
	
	/**
	 * Sets the checkbox to selected.
	 * <br>
	 * buttonNum is between 1 and 4 other numbers are ignored.
	 * 
	 * @param buttonNum The index of the checkbox to be selected (range 1 to 4)
	 */
	public void setSelectedButton(int buttonNum){
		if (buttonNum == 1) {
			buttonOne.setSelected(true);		
		}
		else if (buttonNum == 2) {
			buttonTwo.setSelected(true);
		}
		else if (buttonNum == 3) {
			buttonThree.setSelected(true);
		}
		else if (buttonNum == 4) {
			buttonFour.setSelected(true);
		}
	}
	
	/**
	 * Removes the button and label specified by index if possible.<br>
	 * index is 1 to 4 otherwise ignored.<br><br>
	 * Used for the hint function
	 * 
	 * @param index the index (1 to 4) of the checkbox option to be removed
	 */
	private void removeButtonAndLabel(int index) {
	
		try {
			if (index == 1) {
				this.remove(buttonOne);
				this.remove(buttonIconLabelOne);	
			}
			else if (index == 2) {
				this.remove(buttonTwo);
				this.remove(buttonIconLabelTwo);
			}
			else if (index == 3) {
				this.remove(buttonThree);
				this.remove(buttonIconLabelThree);
			}
			else if (index == 4) {
				this.remove(buttonFour);
				this.remove(buttonIconLabelFour);
			}
		} catch(NullPointerException npe) {/*Something didn't need to be removed - probably a label*/}
		
		this.validate();
		this.repaint();
	}	
	
	/** 
	 * Removes two random buttons and labels from the index's supplied 
	 * in wrongAnswers (wrongAnswers should contain three elements that
	 * reference the wrong answer checkbox locations
	 * 
	 * @param wrongAnswers Vector containing the indexes of the three wrong checkbox locations
	 */
	public void giveHint(Vector wrongAnswers){

		int counter = 0;
		int index1 = 0;
		int index2 = 0;
		Random randNum = new Random();
 		
 		//loop until we get two different indexes (only allow 50 iterations though)
 		//really this is massive overkill as we should always get two different random numbers
 		//it is just in case we don't that we loop around a while 
 		while ((index1 == index2) || (counter > 50)) {
 			index1 = randNum.nextInt(3);//Get a random int between 0,1,2
 			index2 = randNum.nextInt(3);//Get a random int between 0,1,2
 			counter += 1;
 		}
 		if (counter < 50) {//actually have two peusdorandom indexes
			removeButtonAndLabel(((Integer) wrongAnswers.elementAt(index1)).intValue());
			removeButtonAndLabel(((Integer) wrongAnswers.elementAt(index2)).intValue());
		}
		else { //need to fake two 'random' numbers because the computer couldn't give us one
			removeButtonAndLabel(((Integer) wrongAnswers.elementAt(0)).intValue());
			removeButtonAndLabel(((Integer) wrongAnswers.elementAt(1)).intValue());
		}
		resetSelectors();
	
		this.validate();
		this.repaint();
	}
}