import java.awt.*;
import java.applet.*;
import javax.swing.*;

/**
 * MultipleChoiceQuestions.java 1.0 02/12/15
 * <br><br>
 * Multiple Choice Question Applet
 * <br><br>
 * Author       : James McMahon
 * <br><br>
 * This file is the main class that contains all of the 
 * necessary functions to run the applet
 * <br><br>
 * Created      : 22/12/2002
 * <br>
 * Last Updated : 22/12/2002
 */
 
public class MultipleChoiceQuestions extends JApplet {
	
	/** 
	 * Allows access to the external documents that contain the 
	 * startup information, html file urls, and possible answer solutions
	 */ 
	private ExternalDocuments questionAccess; 
	
	/** 
	 * The place for the Question HTML File to be displayed
	 */
	private QuestionPanel questionPanel; 
	
	/** 
	 * The place for the Answer HTML files to be displayed
	 */
	private AnswerPanel   answerPanel;  
	 
	/** 
	 * The possible answercheckboxes (radiobuttons)
	 */
	private AnswerSelectionPanel answerSelections = new AnswerSelectionPanel();
	
	/** 
	 * Keeps track of the user's score for all of the question sets
	 */
	private ScorePanel  scorePanel;
	
	/** 
	 * The Menu panel
	 */
	private MenuPanel   menuPanel  = new MenuPanel(this);
	
	/** 
	 * The content pane of the applet
	 */
	private Container content = getContentPane();
	
	/** 
	 * The whole list of questionSet's recoverable from ExternalDocuments methods
	 */
	private Questions questions;
	
	/** 
	 * The currently loaded question set
	 */
	private QuestionSet questionSet;
	
	/** 
	 * The currently displayed question from the loaded question set
	 */
	private QuestionData questionData;
	
	//Global variables used to stop the users cheating and to make things work more smoothly
	
	/**
	 * true if the user has already submitted a correct answer
	 */
	private boolean submittedCorrectAnswer = false;
	
	/**
	 * true if the user asked for the answer to be revealed
	 */
	private boolean revealedAnswer         = false;
	
	/**
	 * true if the user has asked for a hint already
	 */
	private boolean hinted			       = false;
	
	/** 
	 * The initiliser function that sets up the applet with 
	 * the first question of the first questionset on screen
	 */
	public void init() {
		
		super.init();
	
	//if security restrictions allow access then do this else - output error message
	try {
			//These need to be initilised in here because I'm passing the actual codebase instead of
			//a refernece to the parent class
			questionAccess = new ExternalDocuments(this.getCodeBase(),this.getParameter("startUpFile"));
			questionPanel = new QuestionPanel(this.getCodeBase());
			answerPanel = new AnswerPanel(this.getCodeBase());
		
			//Now that ExternalDocuments has been initilised we can 
			//initilise the questions information for the first set and 
			//first question of that set 
			questions    = questionAccess.getQuestions();
 			questionSet  = questions.getQuestionSet(0);
			questionData = questionSet.getQuestionFromSet(0);
		
			//initlise the score panel with all of the possible set's 
			//and the number of questions in each
			scorePanel = new ScorePanel(questions);
		
			content.setLayout(new BorderLayout());
		
			//Add the question panel that displays the questions to the user on to the screen
			content.add(questionPanel, BorderLayout.NORTH);
		
			//Temporary panel that helps set up a nicer layout
			JPanel centerPanel = new JPanel(new BorderLayout());
		
			//set up the temporary panel so we can have a nicer layout
			centerPanel.add(answerSelections, BorderLayout.NORTH);
			centerPanel.add(answerPanel, BorderLayout.CENTER);
		
			//Add the score panel to the screen
			content.add(scorePanel, BorderLayout.EAST);
			
			//Add the temporary centre panel for a prettier layout
			content.add(centerPanel, BorderLayout.CENTER);
		
			//Add the menuPanel to the screen
			content.add(menuPanel, BorderLayout.SOUTH);	

			//Put the first question of the first question set on to the screen
			//using the previously initilised values
			//N.B. Questions and QuestionSets all start at zero
			setCurrentQuestion(questionData);
			
			scorePanel.initiliseScoring(0);
		} catch(SecurityException e) {
			content.setLayout(new BorderLayout());
			content.add(new JLabel("You need to set security to minimal for untrusted content!", JLabel.CENTER), BorderLayout.NORTH);
			content.add(new JLabel(e.toString()), BorderLayout.CENTER);
		}	
	}

	/** 
	 * Allows the user to select a new question set to try and answer
	 */
	public void selectQuestionSet() {
		
		Object[] questionSetList = questions.getQuestionSetNames();

		String newSet = (String)JOptionPane.showInputDialog(this,
															"Select the question set you want",
															"Question Set Selection",
															JOptionPane.QUESTION_MESSAGE,
															null,
															questionSetList,
															questionSetList[0]);
		if (newSet != null) {//user actually selected something and didn't press cancel
	
			questionSet = questions.getQuestionSet(newSet);//store the new questionSet
				
			questionData = questionSet.getQuestionFromSet(0);//store the current question
			
			setCurrentQuestion(questionData);//show this question on screen
			scorePanel.initiliseScoring(questions.getQuestionSetLocation(questionSet));//initilise the scoring if neccessary
		}

		this.validate();
		this.repaint();
	}

	/** 
	 * Gives the user a hint by removing two of the three wrong answers
	 */
	public void showHint() {
		if (!hinted) {
			answerSelections.giveHint(questionData.getWrongAnswerLocations());
			hinted = true;
			
			this.validate();
			this.repaint();
		}
		else {//user has already had a hint so don't give them another
			JOptionPane.showMessageDialog(this,
 								          "You've already been given a hint!",
 								          "Only one hint per question!",
 								          JOptionPane.INFORMATION_MESSAGE
 								          );
		}
	}

	/** 
	 * Gives the user the answer because they don't want to try and guess it
	 */
	public void showAnswer() {
	
		if (!submittedCorrectAnswer) {	
			//answer has been given so make sure user can't cheat with the answer
			revealedAnswer = true;
			
 			int rightAnswer = this.questionData.getCorrectAnswerLocation();
 			answerPanel.setAnswerPage((String)(questionData.getAllAnswerFileLocations()).elementAt(rightAnswer-1));
 			answerSelections.setSelectedButton(rightAnswer);
 		
 			JOptionPane.showMessageDialog(this,
 										  "Click OK for next question",
 										  "The answer is on display",
 										  JOptionPane.INFORMATION_MESSAGE
 										 );
 		
 			if (questionSet.numberOfQuestionsLeft() != 0) {		
 				setCurrentQuestion(questionSet.getNextQuestionFromSet());//use next question
 			} else {//Not got another question so inform the user to change questionsets
 				JOptionPane.showMessageDialog(this,
 										  	  "Please choose a new question set to try",
 										      "No Questions Left in Current Set",
 										      JOptionPane.INFORMATION_MESSAGE
 										     );
 			}			
 			
 			this.validate();
 			this.repaint();
 		}
 		else {//The user has already answered the question correctly
 			JOptionPane.showMessageDialog(this,
 								  	      "You don't need a hint!",
 								 	      "Already Got The Correct Answer",
 									      JOptionPane.INFORMATION_MESSAGE
 								 		 );
 		}
	}

	/** 
	 * Checks the user's answer for the current question by finding out which checkbox is selected
	 * and whether it is the right one or not
	 */
	public void checkAnswer() {
	
	  if(!revealedAnswer){	
		if(!submittedCorrectAnswer) {	
			
			int rightAnswer = this.questionData.getCorrectAnswerLocation();
 			int selectedAnswer = this.answerSelections.getSelectedAnswer();
 		
 			if (selectedAnswer == 0) {
 				//User hasn't actually selected an answer yet so tell them	
	 			JOptionPane.showMessageDialog(this,
 			  						  	 	  "You need to select an answer first!",
 										      "No Answer Selected",
 										      JOptionPane.ERROR_MESSAGE
 										     );
 			} 
 			else {//Got an answer
 			
 				answerPanel.setAnswerPage((String)(questionData.getAllAnswerFileLocations()).elementAt(selectedAnswer-1));
 			
 				if (selectedAnswer==rightAnswer) {//Got the right answer
 				
 					//record the fact the user's answer was correct
 					submittedCorrectAnswer = true;
 				
 					scorePanel.addOneToScore(questions.getQuestionSetLocation(questionSet));//update score	
 			
 	 				JOptionPane.showMessageDialog(this,
 										  	      "Well Done, click OK for another",
 										          "Correct Answer!",
 										          JOptionPane.INFORMATION_MESSAGE
 										         );		
 			
 					if (questionSet.numberOfQuestionsLeft() != 0) {		
 						setCurrentQuestion(questionSet.getNextQuestionFromSet());//use next question
 					} 
 					else {//Not got another question so inform the user to change questionsets
 						JOptionPane.showMessageDialog(this,
 												  	  "Please choose a new question set to try",
 												      "No Questions Left in Current Set",
 												      JOptionPane.INFORMATION_MESSAGE
 										      	     );
 					}
 				}
 			}
 		} else {//already submitted answer
 			JOptionPane.showMessageDialog(this,
 									  	  "You already know this is correct!?",
 									      "Thinking of Cheating?",
 									      JOptionPane.INFORMATION_MESSAGE
 							      	     );	
 		}
 	  } else {//User is trying to cheat because they've been given the correct answer already
 			JOptionPane.showMessageDialog(this,
 									  	  "Don't you trust my answer?",
 									      "Thinking of Cheating?",
 									      JOptionPane.INFORMATION_MESSAGE
 							      	     );	
 		}
 		
 		this.validate();
 		this.repaint();
	}
	
	/** 
	 * Resets the radiobuttons and answer editor panes to the default states
	 */
	public void resetAllPanels() {
		answerSelections.resetSelectors();
		answerPanel.resetAnswerPanel();
		
		this.validate();
		this.repaint();
	}
	
	/** 
	 * Sets the questiondata to be the currently shown question
	 * <br><br>
	 * Also displays the default answer page.
	 * 
	 * @param questionData the question you want to be displayed on screen
	 */
	public void setCurrentQuestion(QuestionData questionData){

		this.questionData = questionData;
		
		questionPanel.setQuestionPage(questionData.getQuestionFileLocation());
		questionPanel.setQuestionNumber((questionSet.lastQuestionAccessed()+1), questionSet.numberOfQuestionsInSet());
		
		if ((questionData.hasImages()) && (questionData.hasText())){
			answerSelections.setBothSelectors(questionData.getAllCheckBoxesText(), 
											  questionData.getAllImageFileLocations());
		}
		else if((questionData.hasImages()) && !(questionData.hasText())){
			answerSelections.setImageOnlySelectors(questionData.getAllImageFileLocations());
		}
		else if(!(questionData.hasImages()) && (questionData.hasText())){
			answerSelections.setTextOnlySelectors(questionData.getAllCheckBoxesText());
		}
		
		answerPanel.resetAnswerPanel();
		
		//Got new question so reset the global tracker variables
		submittedCorrectAnswer = false;
		revealedAnswer  = false;
		hinted			= false;
		
		this.validate();
		this.repaint();
	}
	
	/**
	 * No explicit threads created by this applet so nothing in particular to do here
	 */
	public void start()	{
		super.start();
	}
	
	/**
	 * No explicit threads created by this applet so nothing in particular to do here
	 */
	public void stop()	{
		super.stop();
	}	
}