 import java.util.*;
 
 /**
 * MultipleChoiceQuestions.java 1.0 22/12/12
 * <br><br>
 * Author       : James McMahon
 * <br><br>
 * Encapsulates the questionData for each QuestionSet
 * <br>
 * This allows access to each question of the questionset 
 * while QuestionData allows access to the specific question details 
 * <br><br>
 * Created      : 22/12/2002 
 * <br>
 * Last Updated : 22/12/2002 
 */
 public class QuestionSet {
 	
 	/** 
 	 * The name of the questionSet
 	 */
 	private String questionSetName = null;
 	
 	/** 
 	 * Stores all the questions associated with this set
 	 */
 	private Vector questions = new Vector(5);
 	
    /**
 	 * The number of questions in the set
 	 */
 	private int numberOfQuestionsAvailable = 0;
 	
 	/** 
 	 * Keeps track of the last question that was accessed (and should have been shown)
 	 */
 	private int lastQuestionAccessed = 0;
 	
 	/**
 	 * Creates a question set without any questions called name
 	 * @param name The name of the question set
 	 */
 	public QuestionSet(String name){
 		questionSetName = name;
 	}	
 	
 	/** 
 	 * Add a question to this set of questions
 	 * 
 	 * @param question The question data to be added to this set
 	 */
 	public void addQuestionToSet(QuestionData question){
 		numberOfQuestionsAvailable += 1;
 		this.questions.addElement(question);	
 	}
 	
 	/**  
 	 * Get a specific question at location from this set of questions
 	 * 
 	 * @param location The location that you want to access (out of bounds location returns last possible location)
 	 * @return QuestionData The data found at that location
 	 */
 	public QuestionData getQuestionFromSet(int location){
 		this.lastQuestionAccessed = location;
 		QuestionData qData = (QuestionData) this.questions.elementAt(location);
 		qData.randomiseAnswerLocations();
 		return qData;
 	}
 	
 	/** 
 	 * Get the name of this question set
 	 * 
 	 * @return String The question set name
 	 */
 	public String getQuestionSetName(){
 		return this.questionSetName;
 	}
 	
 	/** 
 	 * Returns the next question in the logical sequence of the question set.
 	 * <br><br>
 	 * This does not currently check to see if there is another element to access.
 	 * <br><br> 
 	 * So be careful when using this function - It will always return the last question in the 
 	 * set if you call it more times than necessary to get the available questions out.
 	 *
 	 * @return QuestionData The next question in the sequence from the question set
 	 */ 
 	public QuestionData getNextQuestionFromSet(){
 		
 		QuestionData qData;
 		
 		if (this.lastQuestionAccessed < this.questions.size()) {
 			this.lastQuestionAccessed++;
 			qData = (QuestionData) this.questions.elementAt(lastQuestionAccessed);
 		} 
 		else {
 			this.lastQuestionAccessed = this.numberOfQuestionsAvailable;
 			qData = (QuestionData) this.questions.lastElement();	

 		}
 		qData.randomiseAnswerLocations();//makes sure the answers are re-positioned
 		return qData;
 	}
 	
 	/** 
 	 * Gets the total number of questions available
 	 * @return int The number of questions available in the set
 	 */
 	public int numberOfQuestionsInSet() {
 		return numberOfQuestionsAvailable;
 	}
 	
 	/** 
 	 * The number of questions that are left to be accessed
 	 * 
 	 * @return int The number of unasked questions left in the set 
 	 */
 	public int numberOfQuestionsLeft() {
 		//as last question accessed runs from 0 - n we need to minus an extra one for the right answer
 		return (numberOfQuestionsAvailable-lastQuestionAccessed-1);
 	}
 	
 	/**
 	 * Accessor for the lastQuestionAsked field 
 	 *
 	 * @return int The index of the last question that was asked
 	 */
 	public int lastQuestionAccessed() {
 		return lastQuestionAccessed;
 	}
 	
 	/**
 	 * Helper function for the question randomise function.<br>
 	 * Does the actual randomisation of questions
 	 * 
 	 * @param questionsToRandomise The list of questions to randomise
 	 * @return Vector A randomised list
 	 */
 	private Vector randomiseQuestionLocations(Vector questionsToRandomise){
 		
 		Vector randomOrder = new Vector();
 		Random randNum = new Random();
 		
 		while(questionsToRandomise.size() != 0) {
 			//Get a random int between 0,1,2 - (size-1)
 			int location = randNum.nextInt(questionsToRandomise.size());
 			randomOrder.addElement(questionsToRandomise.elementAt(location));
 			questionsToRandomise.removeElementAt(location);
 		}
 	
 		return randomOrder;
 	}
 	
 	/**  
 	 * Randomise the locations of questions within the set
 	 */
 	public void randomise(){
 		this.questions = randomiseQuestionLocations(this.questions);
 	}
 }