import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import javax.swing.*;
import javax.swing.event.*;
import java.io.*;
import java.util.*;
import java.net.*;

/**
 * MultipleChoiceQuestions.java 1.0 22/12/12
 * <br><br>
 * Author       : James McMahon
 * <br><br>
 * Provides the ScorePanel that displays the scores for all question sets
 * <br><br>
 * It can be set dynamically from the main class based on which questionset 
 * is being updated. It needs to be initilised with the exact number of 
 * question sets that are available.
 * <br><br>
 * N.B. The number of Question Sets cannot exceed 20 although there can 
 * be as many questions as you like in each question set
 * <br><br>
 * Created      : 22/12/2002
 * <br>
 * Last Updated : 22/12/2002
 */

public class ScorePanel extends JPanel{

	/** 
	 * Each entry in scoresArray contains the number of answers that the user got right
	 */
	private int scoresArray[];
	
	/** 
	 * Each entry in numQuestionsArray contains the total number of questions that
	 * are availble in each question set
	 */
	private int numQuestionsArray[];

	/** 
	 * The panel that the labels are stuck on to 
	 */
	private JPanel labelPanel;// = new JPanel(new FlowLayout());
	
	/** 
	 * The ScrollPane the score label panel gets added to 
	 */
	private JScrollPane scrollPane;
	
	
	/** 
	 * Constructs the score panel based on the questions sets that have been loaded
	 * 
	 * @param availableQuestionSets The loaded sets so that the numbers of questions can be displayed
	 */
	public ScorePanel(Questions availableQuestionSets) {
	
		this.setLayout(new BorderLayout());
		this.setPreferredSize(new Dimension(160,110));
		this.setBorder(BorderFactory.createTitledBorder("Score Tracker"));

		int numQuestionSets = availableQuestionSets.numberOfQuestionSets();
	
		//Initlise these to hold as many question sets as needed
		scoresArray 	  = new int[numQuestionSets];
		numQuestionsArray = new int[numQuestionSets];
	
		int counter = 0;
		
		//initilise everything in the array's
		while( counter < numQuestionSets){
			scoresArray[counter] = -1;
			numQuestionsArray[counter] = (((availableQuestionSets).getQuestionSet(counter)).numberOfQuestionsInSet());
			counter++;
		}
		
		this.add(new JLabel("Question Sets Scores"), BorderLayout.NORTH);
		
		//the current question set is defaulted in main class to the firstone
		displayLabels(0);
	}	

	/** 
	 * Display all of the information in the arrays on to the screen
	 * 
	 * @param currentQuestionSet The questionset label that is to be 
	 *                           highlighted, should be the one the user 
	 *                           is currently editing.
	 */
	public void displayLabels(int currentQuestionSet){

		this.labelPanel = new JPanel(new FlowLayout());
		this.labelPanel.setPreferredSize(new Dimension(140,90));
		
		for(int i=0;i<scoresArray.length;i++){

			if(scoresArray[i] != -1){
				if (currentQuestionSet == i) {//This is the current questionset so make the text bold
					JLabel label = new JLabel("Set "+(i+1)+": "+scoresArray[i]+" out of "+numQuestionsArray[i]);
						   Font font = new Font("SansSerif", Font.BOLD|Font.ITALIC, 15);
						   label.setFont(font);
					labelPanel.add(label);
				}
				else {//not current so not bold
					labelPanel.add(new JLabel("Set "+(i+1)+": "+scoresArray[i]+" out of "+numQuestionsArray[i]));
				}
			}
			else{//the user hasn't started answering this question set yet
				labelPanel.add(new JLabel("Set "+(i+1)+": - out of "+numQuestionsArray[i]), BorderLayout.SOUTH);
			}
		}
		
		try{
			this.remove(scrollPane);
		}catch(NullPointerException npe){/*no need to remove because it wasn't there to start with*/}
		scrollPane = new JScrollPane(labelPanel);
		
		this.add(scrollPane, BorderLayout.CENTER);
		
		//For some reason it stops displaying it's self properly so we need to force it
		this.setVisible(true);
		this.scrollPane.setVisible(true);
		this.validate();
		this.repaint();
	}

	/** 
	 * add one on to the score of set index, 0-numPossibleSets
	 *
	 * @param index The index of the question set that you want to add one to score of
	 */
	public void addOneToScore(int index) {
		
		this.scoresArray[index] += 1;
		displayLabels(index);//Update the display on screen
	}

	/**
	 * This method is required because before starting a question set score is actually negative.
	 * <br><br>
	 * Hence this method set's the score to zero in the case that it hasn't yet been initilised already.
	 * <br><br>
	 * Otherwise it set's the score to zero because we assume a previously accessed question set has been reaccessed.
	 * <br><br>
	 * This is usually called when a user loads a new questionset
	 * 
	 * @param index The index of the question set that is being initilised
	 */
	public void initiliseScoring(int index){
		if (scoresArray[index] == -1) {
			addOneToScore(index);
		}
		else {
			scoresArray[index] = 0;
			displayLabels(index);
		}
	}
}